/*  *********************************************************
	*           PVX PLUS PROPRIETARY & SENSITIVE            *
	* ----------------------------------------------------- *
	*  The information contained herein is the exclusive    *
	*  property of PVX Plus Technologies Ltd (Ontario).     *
	*  This material shall not be reproduced, or its        *
	*  contents disclosed to others in whole or in part.    *
	* ----------------------------------------------------- *
	*   Copyright (c) 2012-2014                             *
	*            PVX Plus Technologies (Ontario, Canada)    *
	*********************************************************

	Interface for the PxIO library. */

#ifndef _PXIO_H_
#define _PXIO_H_

#include <stddef.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/*******************************************************************************
	Macros & Types
*******************************************************************************/

/*  When building a Windows DLL we want to use the __stdcall calling convention
    so as to have maximum compatability with various DLL consumers. For example
    visual basic assumes the __stdcall callign convention for DLL's. If not
    building a DLL then use the default calling convention. */
#if !defined (PXIO_STATIC) && defined (WIN32)
#define APIDEF __stdcall
#else
#define APIDEF
#endif

/*  GCC versions 4 and higher support specifying which library
	functions are visible to users of the dynamic library. This is useful as it
	avoids making internal functions visable and it produces more optimized
	code. For Windows functions are exported through the DEF file. */
#if !defined (PXIO_STATIC) && !defined (WIN32) && (__GNUC__ >= 4)
#define PXIO_PUBLIC __attribute__ ((visibility ("default")))
#else
#define PXIO_PUBLIC
#endif

/*  We do this instead of include bool_type.h so that we do not have to
	distribute more than one header with our library. */
#ifndef BOOL_TYPE_H
#define BOOL_TYPE_H

typedef int BOOL;

#ifndef FALSE
#define FALSE 0
#endif

#ifndef TRUE
#define TRUE 1
#endif

#endif

/*  PxIOGetError return value if passed in return status was a non error status.
	Used this in conjunction with PxIOGetError to check if a PxIO library call
	returned successfully. */
#define PXIO_NO_ERROR -1

/*  The function returned successfully. */
#define PXIO_SUCCESS_STATUS 1

/*  Stub type for a service. */
typedef uint32_t PxIOService;

/*  Stub type for file handle. */
typedef uint32_t PxIOFileHandle;

/*  Options for use with 'flags' argument of PxIOOpen. */
#define OPEN_LOCK       0x00000001 /* Lock file for exclusive use. */
#define OPEN_LOAD       0x00000002 /* Load file into memory. File cannot be written to. */
#define OPEN_READ_ONLY  0x00000004 /* Open as Read Only. */
#define OPEN_PURGE      0x00000008 /* Purge the file after it's opened. */
#define OPEN_APPEND     0x00000010 /* Append all writes to the end of the file. Only applicable to Sequential and Binary files. */
#define OPEN_BBX        0x00000020 /* Open file in BBx mode. */
#define OPEN_CREATE     0x00000040 /* Not implemented. */

/*  Key type used to define which key to retrieve. */
typedef enum PxIOKeyType
{
	KEY_TYPE_FIRST,     /*  The first key in the file. */
	KEY_TYPE_CURRENT,   /*  The current key. */
	KEY_TYPE_NEXT,      /*  The key after the current key. */
	KEY_TYPE_PREVIOUS,  /*  The key before the current key. */
	KEY_TYPE_NEXT_NEXT, /*  The key two after the current key. */
	KEY_TYPE_LAST       /*  The last key in the file. */
} PxIOKeyType;

/*  Read type is used to tell the read function how you want to read the data. */
typedef enum PxIOReadType
{
	READ_TYPE_NO_OPTIONS    = 0x00, /*  No special options required for read */
	READ_TYPE_IGNORE_LOCK   = 0x01, /*  Ignore any locks on the record. */
	READ_TYPE_LOCK          = 0x02, /*  Lock the record before reading. */
	READ_TYPE_FIND          = 0x04, /*  If record is not found do not change the file position. */
} PxIOReadType;

/*  Type used to specify information returned by PxIOFileInfo function. */
typedef enum PxIOFileInfoType
{
	INFO_TYPE_EXTERNAL_KEY_SIZE,  /*  Key size */
	INFO_TYPE_RECORD_SIZE,        /*  Record size */ 
	INFO_TYPE_RECORD_COUNT,       /*  Record count */ 
	INFO_TYPE_FILE_SIZE,          /*  File size in bytes */
	INFO_TYPE_RECORD_MAX,         /*  Maximum # of records */
	INFO_TYPE_FILE_TYPE,          /*  File type. See table in PxIOFileCreate. */
	INFO_TYPE_FIELD_SEPARATOR,    /*  The field separator used in the file. */
	INFO_TYPE_BLOCK_THRESHOLD,    /*  Block usage threshold. */
	INFO_TYPE_BLOCK_SIZE,         /*  Block size in kilobytes. */
	INFO_TYPE_CREATION_OPTIONS,   /*  Options used when the file was created. */
	INFO_TYPE_KEY_SEQUENCE_COUNT, /*  Number of key streams. */
	INFO_TYPE_LOCK_SIZE,          /*  Size of record lock. */
	INFO_TYPE_CREATION_STAMP,     /*  Internal file header creation stamp. */
	INFO_TYPE_ATTRIBUTES          /*  File attributes. */
} PxIOFileInfoType;

/*  Type used to specify the type of file to create. */
typedef enum PxIOFileType
{
	FILE_TYPE_SEQUENTIAL,   /*  Sequential file */
	FILE_TYPE_KEYED,        /*  Standard keyed/sort */
	FILE_TYPE_INDEXED,      /*  Indexed/Relative */
	FILE_TYPE_BINARY,       /*  Binary file */
	FILE_TYPE_DIRECTORY,    /*  Directory */
	FILE_TYPE_ODBC,         /*  ODBC Access */
	FILE_TYPE_OCI,          /*  OCI File */
	FILE_TYPE_DLL,          /*  DLL file */
	FILE_TYPE_KY2,          /*  KY2 File */
	FILE_TYPE_PDF,          /*  PDF File */
	FILE_TYPE_DB2,          /*  DB2 Access */
	FILE_TYPE_MYSQL_PLUS,   /*  PvxPlus - MYSQL Direct access */
	FILE_TYPE_ADO,          /*  ADO interface */
    FILE_TYPE_ZIP           /*  Zip file */
} PxIOFileType;

/*  Defines used to specify dictionary returned by PxIODictionaryInfo. */
typedef enum PxIODictionaryInfoType
{
	DICTIONARY_INFO_TYPE_PRIMARY,    /*  Return the primary dictionary */
	DICTIONARY_INFO_TYPE_ALTERNATE,  /*  Return the alternate dictionary */
	DICTIONARY_INFO_TYPE_FORMATS,    /*  Return valid record formats */
	DICTIONARY_INFO_TYPE_EMBEDDED    /*  Return the embedded io path */
} PxIODictionaryInfoType;

/*  Options for use with 'options' argument of PxIOFileCreate. */
#define CREATE_OPTION_COMPRESS       0x00000001 /* Add simple compression to record data. */
#define CREATE_OPTION_VARRECSZ       0x00000002 /* Variable length records (FILE_TYPE_KEYED only). */
#define CREATE_OPTION_EXTENDED       0x00000004 /* Extended Record Size. */
#define CREATE_OPTION_ZLIBCMP        0x00000008 /* Set ZLib Compression for VLR and EFF Files. */
#define CREATE_OPTION_UPDATEPLUS     0x00000010 /* Enable UpdatePlus logic (FILE_TYPE_KEYED only). */
#define CREATE_OPTION_EFFUNLIMITED   0x00000020 /* No 2GB limit (FILE_TYPE_KY2 only). */
#define CREATE_OPTION_NO_SEPARATOR   0x00000040 /* No field separator. */
#define CREATE_OPTION_BBX_SEPARATORS 0x00000080 /* Use BBX field separators. */
#define CREATE_OPTION_NULL_KEY_STRIP 0x00000100 /* BBX style NULL Keys (strips trailing nulls from a key). */

/*  Flags set by PxIOGetFileInfo in the fileInfo parameter when the INFO_TYPE_ATTRIBUTES argument type is passed. */
#define ATTRIBUTE_EXTENDED_KEY_SEG   0x00000001 
#define ATTRIBUTE_REBUILDING_KEY     0x00000002 
#define ATTRIBUTE_FIXED_KEY_TABLE    0x00000004 
#define ATTRIBUTE_JOURNALIZE         0x00000008 

/*  Type used to specify the library option to get/set. */
typedef enum PxIOLibOptionType
{
	LIB_OPTION_TYPE_IGNORE_MAX_REC,         /*  Ignore maxmimum record count errors (Default: false) */
	LIB_OPTION_TYPE_SERIAL_NO_LOCK,         /*  Serial files don't need to be locked (Default: false) */
	LIB_OPTION_TYPE_SHRINK_KEYED_FILE,      /*  Shrink keyed files (Default: false) */
	LIB_OPTION_TYPE_VLR_SEGMENTS_MODIFY,    /*  Allow VLR segment modification (Default: false) */
	LIB_OPTION_TYPE_DEDICATED_BUFFER,       /*  Number of dedicated file buffers (Default: 5) */
	LIB_OPTION_TYPE_SEGMENT_SIZE,           /*  File segment size in MB (Default: 0) */
	LIB_OPTION_TYPE_EXCLUSIVE_FILE_USE      /*  Files are opened for exclusive library use. */
} PxIOLibOptionType;

/*  Type used to specify the service option to get/set. */
typedef enum PxIOServiceOptionType
{
	SERVICE_OPTION_TYPE_KEYED_FILE_BBX,         /*  Keyed File IO Emulates BBX (Default: false) */
} PxIOServiceOptionType;

/*  Type used to specify the type of file time to retrieve or set. */
typedef enum PxIOFileTimeType
{
	FILE_TIME_TYPE_CREATION_TIME,
	FILE_TIME_TYPE_MODIFICATION_TIME,
	FILE_TIME_TYPE_ACCESS_TIME
} PxIOFileTimeType;

/*  Options for use with the 'permission' argument of PxIOPasswordSet. */
#define POPEN      0x00 /* Password required for open access. */
#define PWRITE     0x01 /* Password required for write access. */
#define POPENDATA  0x02 /* Password always required, data encrypted. */
#define PWRITEDATA 0x03 /* Password required for write access, data encrypted. */

/*  Structure used to pass back record location and size info by several functions. */
typedef struct RecordInfo
{
	int             length; /* Record length */
	char            *data;  /* Record data pointer */
} RecordInfo;

/*  Structure used to pass back record location and size info by several functions. */
typedef struct KeyInfo
{
	int             length; /* Key length */
	char            *data;  /* Key data pointer */
} KeyInfo;

/*******************************************************************************
	Prototypes
*******************************************************************************/
/*----------------------------------------------------------------------------*/
/*  Error Functions
------------------------------------------------------------------------------*/
/*
	PxIOGetError:
	Given a return status convert it to an error code if applicable (i.e. a negative
	number). The function will also look up and return the error message if
	applicable and passed a buffer.

	If the passed in return status was not an error status PXIO_NO_ERROR is
	returned.
*/
PXIO_PUBLIC int APIDEF PxIOGetError(int returnStatus, char *errorMessageBuffer, size_t *errorMessageBufferSize);
/*
	Parameters:
		returnStatus       -The return status to convert.
		errorMessageBuffer -A string buffer to store an error message.
		errorMessageBuffer -On input the size of the string buffer, on output
		                    the length of the error message or the needed
		                    buffer size if there was a length error.

	Return:
		The converted error code on success, an error status on failure. 

------------------------------------------------------------------------------*/
/*  Global Library Functions
------------------------------------------------------------------------------*/
/*
	PxIOActivation:
	Activate the PxIO library. This must be called prior to opening
	a file in order to provide the library with a valid activation. Without
	activation, a warning message that requires user intervention will be
	displayed whenever any file is opened.
*/
PXIO_PUBLIC int APIDEF PxIOActivation(const char *activationString, int activationNumber);
/*
	Parameters:
		activationString -Registration string provided by PVX Plus Technologies Ltd.
		activationNumber -Registration number provided by PVX Plus Technologies Ltd.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOLibInit:
	Initializes the PxIO library. This function MUST be the first function called,
	and must be called before making any other calls to the library.
*/
PXIO_PUBLIC int APIDEF PxIOLibInit(BOOL multithread);
/*
	Parameters:
		multithread -Used to indicate if the library will be expected to support a multithreading application.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOLibShutDown:
	Shuts down the PxIO library. All services should be terminated before calling
	this function. All allocated resources are return to the operating system.
	After calling this function, no further calls will be serviced, including
	PxIOLibInit.
*/
PXIO_PUBLIC int APIDEF PxIOLibShutDown(void);
/*
	Parameters: None.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOGetLibOption:
	Return the current value of a library option.
*/
PXIO_PUBLIC int APIDEF PxIOGetLibOption(PxIOService service, PxIOLibOptionType type, int *value);
/*
	Parameters:
		service -The handle to a PxIOService associated with the library.
		type    -The option to retrieve.
		value   -The storage used to save the option value.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOSetLibOption:
	Modify a library option.
*/
PXIO_PUBLIC int APIDEF PxIOSetLibOption(PxIOService service, PxIOLibOptionType type, int value);
/*
	Parameters:
		service -The handle to a PxIOService associated with the library
		type    -The option to set.
		value   -The value to set the option with.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*  Service Functions
------------------------------------------------------------------------------*/
/*
	PxIOCreateService:
	Creates and initializes a PxIOService object used for interfacing with the
	library. If a host name/IP address and port are specified, a connection
	will be made to PxServer and library calls using this service will be
	executed remotely on PxServer. 
*/
PXIO_PUBLIC int APIDEF PxIOCreateService(PxIOService *service, const char *hostname, int port,
	BOOL sslFlag, const char *caFile, const char *userID, const char *companyCode);
/*
	Parameters:
		service     -Pointer to storage to save the newly created PxIOService.
		hostname    -If connecting to a PxServer supply an ip address, if not
		             pass in NULL.
		port        -If connecting to a PxServer supply a port number, if not
		             pass in zero.
		sslFlag     -TRUE if SSL should be used to connect to the server, FALSE
		             otherwise.
		caFile      -Filename path of the certificate authority file, or NULL if
		             SSL or server trust verification is not required.
		UserId      -If connecting to a PxServer supply a user id for security
		             verification, if not pass in NULL.
		companyCode -If connecting to a PxServer supply a company code for
		             security verification, if not pass in NULL.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIODestroyService:
	This will destroy the PxIOService and should be used when you are done
	doing library IO.
*/
PXIO_PUBLIC int APIDEF PxIODestroyService(PxIOService service);
/*
	Parameters:
		service -The PxIOService to destroy.

------------------------------------------------------------------------------*/
/*
	PxIOGetServiceOption:
	Return the current value of a service option.
*/
PXIO_PUBLIC int APIDEF PxIOGetServiceOption(PxIOService service, PxIOServiceOptionType type, int *value);
/*
	Parameters:
		service -The PxIOService to get the option from.
		type    -The option to retreive.
		value   -The storage used to save the service option value.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOSetServiceOption:
	Modify a service option.
*/
PXIO_PUBLIC int APIDEF PxIOSetServiceOption(PxIOService service, PxIOServiceOptionType type, int value);
/*
	Parameters:
		service -The PxIOService where the option is set.
		type    -The service option to set.
		value   -The value to set the service option with.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*  File Open and Close Functions
------------------------------------------------------------------------------*/
/*
	PxIOOpen:
	Opens a PxPlus type file locally or remotely depending on the service passed
	in.
*/
PXIO_PUBLIC int APIDEF PxIOOpen(PxIOService service, PxIOFileHandle *handle, const char *pathname, int16_t isz, const char *password, const char *options, int flags);
/*
	Parameters:
		service  -The PxIOService to use when opening the file.
		handle   -Pointer to storage to save the newly created PxIOFileHandle.
		pathname -The pathname of the file to open. If remote this should be
		          relative to the remote system.
		isz      -Desired record size for binary mode.
		password -The file password if any. NULL terminated.
		options  -Used only for devices?
		flags    -A bitmap indicating how to open the file.

	Return:
		A success status on success, an error status on failure. 

------------------------------------------------------------------------------*/
/*
	PxIOClose:
	Closes a file.
*/
PXIO_PUBLIC int APIDEF PxIOClose(PxIOFileHandle handle);
/*
	Parameters:
		handle -The handle to the file to close.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*  File Information Functions
------------------------------------------------------------------------------*/
/*
	PxIOGetFileInfo:
	Returns the requested file information. See earlier definition of
	PxIOFileInfoType for the file information this function returns.
*/
PXIO_PUBLIC int APIDEF PxIOGetFileInfo(PxIOFileHandle handle, PxIOFileInfoType type, int *fileInfo);
/*
	Parameters:
		handle   -The handle to the file to retrieve the file info from.
		type     -The type of file information requested.
		fileInfo -Storage used to save the reteived file information.

	Return:
		A success status on success, an error status on failure. 

------------------------------------------------------------------------------*/
/*
	PxIOGetKeyLayout:
	Return the key layout.
*/
PXIO_PUBLIC int APIDEF PxIOGetKeyLayout(PxIOFileHandle handle, char *keyLayoutBuffer, size_t *keyLayoutBufferSize);
/*
	Parameters:
		handle              -The handle to the file to retrieve the key layout from.
		keyLayoutBuffer     -The buffer in which to return the key layout.
		keyLayoutBufferSize -On input the size of the key layout buffer, on output
		                     the length of the key info or the needed buffer
		                     size if there was a length error.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOGetRawKeyLayout:
	Return the raw binary key layout.
*/
PXIO_PUBLIC int APIDEF PxIOGetRawKeyLayout(PxIOFileHandle handle, char *rawKeyLayoutBuffer, size_t *rawKeyLayoutBufferSize);
/*
	Parameters:
		handle                 -The handle to the file to retrieve the raw key layout from.
		rawKeyLayoutBuffer     -The buffer in which to return the raw key layout.
		rawKeyLayoutBufferSize -On input the size of the raw key layout buffer, on output
		                        the length of the raw key info or the needed buffer
		                        size if there was a length error.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*  File Position Functions
------------------------------------------------------------------------------*/
/*
	PxIOKeySeek:
	Positions the key pointer to a specified location within a file.
*/
PXIO_PUBLIC int APIDEF PxIOKeySeek(PxIOFileHandle handle, KeyInfo *key, int keyNumber);
/*
	Parameters:
		handle    -The handle to the file to perform the seek on.
		key       -Pointer to a keyInfo structure containing information for
		           the key to seek to.
		keyNumber -The key number used to specify whether the key is the
		           primary or one of the alternate keys (-1 for current?).

	Return:
		A success status on success, an error status on failure. 

------------------------------------------------------------------------------*/
/*
	PxIOGetPosition:
	Saves the current record number of the specified file in the 'position'
	parameter.
*/
PXIO_PUBLIC int APIDEF PxIOGetPosition(PxIOFileHandle handle, uint64_t *position);
/*
	Parameters:
		handle   -The handle to the file to get the position of.
		position -The storage used to save the position.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOGetNextPosition:
	Saves the current record number of the specified file in the 'position'
	parameter.
*/
PXIO_PUBLIC int APIDEF PxIOGetNextPosition(PxIOFileHandle handle, int keyNumber, uint64_t *position);
/*
	Parameters:
		handle    -The handle to the file to get the position of.
		keyNumber -The key number used to specify whether the key is the
		           primary or one of the alternate keys (-1 for current?).
		position  -The storage used to save the position.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOGetKeyPosition:
	Saves the index number of the specified key in the 'position' parameter.
*/
PXIO_PUBLIC int APIDEF PxIOGetKeyPosition(PxIOFileHandle handle, KeyInfo *key, int keyNumber, uint64_t *position);
/*
	Parameters:
		handle    -The handle to the file to get the position of.
		key       -Pointer to a keyInfo structure containing key information used as the basis to determine the index value.
		keyNumber -The key number used to specify whether the key is the
		           primary or one of the alternate keys (-1 for current?).
		position  -The storage used to save the index.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOPositionSeek:
	Sets the offset pointer of the specified file to the requested position.
*/
PXIO_PUBLIC int APIDEF PxIOPositionSeek(PxIOFileHandle handle, uint64_t position);
/*
	Parameters:
		handle   -The handle to the file to perform the seek on.
		position -The position to position the file to (position being the address).

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOGetRecordNumber:
	Get the logical sequence number of the current record for an index file.
*/
PXIO_PUBLIC int APIDEF PxIOGetRecordNumber(PxIOFileHandle handle, int *recordNumber, int keyNumber);
/*
	Parameters:
		handle      -The handle to the file to get the index of.
		recordNumber -The storage used to save the index.
		keyNumber   -Key number the record sequence is based upon.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*  File Read Functions
------------------------------------------------------------------------------*/
/*
	PxIORead:
	Read the next record based on the current file offset. Automatically
	increments the file offset to the next entry.
*/
PXIO_PUBLIC int APIDEF PxIORead(PxIOFileHandle handle, RecordInfo *record, PxIOReadType readType);
/*
	Parameters:
		handle   -The handle to the file to read from.
		record   -On input an empty RecordInfo structure, on output it will contain
		          a buffer containing requested record contents and the buffer
		          size.
		readType -Option to use when reading. See the earlier definition of
		          PxIOReadType for more information.

	Return:
		A success status on success, an error status on failure. 

------------------------------------------------------------------------------*/
/*
	PxIOKeyRead:
	Read the record associated with the given key from the specified file.
*/
PXIO_PUBLIC int APIDEF PxIOKeyRead(PxIOFileHandle handle, RecordInfo *record, KeyInfo *key, int keyNumber, PxIOReadType readType);
/*
	Parameters:
		handle    -The handle to the file to read from.
		record    -On input, an empty RecordInfo structure. On output it will
		           contain a pointer to a buffer containing requested record
		           contents, and the buffer size.
		key       -Pointer to a keyInfo structure containing information for
		           the key of the record to be read.
		keyNumber -The key number used to specify whether the key is the
		           primary or one of the alternate keys (-1 for current?). <<<<<<<<
		readType  -Option to use when reading. See the earlier definition of
		           PxIOReadType for more information.

	Return:
		A success status on success, an error status on failure. 

------------------------------------------------------------------------------*/
/*
	PxIOIndexRead:
	Read the record associated with the given index number from the specified file.
*/
PXIO_PUBLIC int APIDEF PxIOIndexRead(PxIOFileHandle handle, struct RecordInfo *record, uint64_t indexNumber, PxIOReadType readType);
/*
	Parameters:
		handle      -The handle to the file to read from.
		record      -On input an empty RecordInfo structure, on output it will contain
		             a buffer containing requested record contents and the buffer
		             size.
		indexNumber -The index of the record to read.
		readType    -Option to use when reading. See the earlier definition of
		             PxIOReadType for more information.

	Return:
		Record size on success, an error status on failure. 

------------------------------------------------------------------------------*/
/*
	PxIORecordRead:
	Read the record associated with the given record number and key number
	parameters from the specified file.
*/
PXIO_PUBLIC int APIDEF PxIORecordRead(PxIOFileHandle handle, RecordInfo *record, int recordNumber, int keyNumber, PxIOReadType readType);
/*
	Parameters:
		handle       -The handle to the file to read from.
		record       -On input an empty RecordInfo structure, on output it will contain
		              a buffer containing requested record contents and the buffer
		              size.
		recordNumber -The number of the record to read.
		keyNumber    -Specifies the key sequence that recordNumber is based upon. 
		readType     -Option to use when reading. See the earlier definition of
		              PxIOReadType for more information.

	Return:
		Record size on success, an error status on failure. 

------------------------------------------------------------------------------*/
/*
	PxIOGetKey:
	Get a key from the specified file.
*/
PXIO_PUBLIC int APIDEF PxIOGetKey(PxIOFileHandle handle, KeyInfo *key, int keyNumber, PxIOKeyType keyType);
/*
	Parameters:
		handle    -The handle to the file to get the key from.
	    key       -Pointer to a keyInfo structure. When the function is
		           called, the data field must be a pointer to a buffer to
		           receive the key information, and the length field must
		           contain the size of the buffer. Upon return, the length
		           field will contain the size of the returned key, or the
		           needed buffer size if there was a length error.
		keyNumber -The key number used to specify whether to get the
		           primary or one of the alternate keys (-1 for current?).
		keyType   -What key to retrieve.

	Return:
		A success status on success, an error status on failure. 

------------------------------------------------------------------------------*/
/*  File Write Functions
------------------------------------------------------------------------------*/
/*
	PxIOWrite:
	Write record to the specified file.
*/
PXIO_PUBLIC int APIDEF PxIOWrite(PxIOFileHandle handle, RecordInfo *record, BOOL overwrite);
/*
	Parameters:
		handle    -The handle to the file to perform the write on.
		record    -A pointer to a RecordInfo structure containing information on
		           the record to be written.
		overwrite -This flag indicates whether the record passed should replace a
		           previously existing record if adding the new record would result
		           in a key violation in one of the file's key sequences (if any).

	Return:
		A success status on success, an error status on failure. 

------------------------------------------------------------------------------*/
/*
	PxIOKeyWrite:
	Write/Rewrite record with an external key to the specified file.
*/
PXIO_PUBLIC int APIDEF PxIOKeyWrite(PxIOFileHandle handle, RecordInfo *record, KeyInfo *key, BOOL overwrite);
/*
	Parameters:
		handle    -The handle to the file to perform the write on.
		record    -A pointer to a RecordInfo structure containing information on
		           the record to be written.
		key       -A pointer to a KeyInfo structure containing information on the
		           key to be written.
		overwrite -This flag indicates whether the information passed should replace
		           previously existing information if adding the new information would
		           result in a key violation in one of the file's key sequences.

	Return:
		A success status on success, an error status on failure. 

------------------------------------------------------------------------------*/
/*
	PxIOIndexWrite:
	Write/Rewrite record with an index to the specified file.
*/
PXIO_PUBLIC int APIDEF PxIOIndexWrite(PxIOFileHandle handle, RecordInfo *record, uint64_t indexNumber, BOOL overwrite);
/*
	Parameters:
		handle      -The handle to the file to perform the write on.
		record      -A pointer to a RecordInfo structure containing information on
		             the record to be written.
		indexNumber -The index number to write the record to.
		overwrite   -This flag indicates whether the record passed should replace a
		             previously existing record if adding the new record would result
		             in a key violation in one of the file's key sequences (if any).

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*  File Remove Functions
------------------------------------------------------------------------------*/
/*
	PxIORemove:
	Removes the current record from the specified keyed file.
*/
PXIO_PUBLIC int APIDEF PxIORemove(PxIOFileHandle handle);
/*
	Parameters:
		handle -The handle to the file to remove the record from.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOKeyRemove:
	Remove a record from the specified keyed file.
*/
PXIO_PUBLIC int APIDEF PxIOKeyRemove(PxIOFileHandle handle, KeyInfo *key);
/*
	Parameters:
		handle -The handle to the file to remove the record from.
		key    -Pointer to a keyInfo structure containing information for the
		        key of the record to be removed.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOIndexRemove:

	Remove a record from the specified keyed file based on the record's index
	number.
*/
PXIO_PUBLIC int APIDEF PxIOIndexRemove(PxIOFileHandle handle, uint64_t indexNumber);
/*
	Parameters:
		handle      -The handle to the file to remove the record from.
		indexNumber -Index number of the record to be removed.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIORecordRemove:

	Remove a record from the specified keyed file based on the record number and
	key number.
*/
PXIO_PUBLIC int APIDEF PxIORecordRemove(PxIOFileHandle handle, int recordNumber, int keyNumber);
/*
	Parameters:
		handle       -The handle to the file to remove the record from.
		recordNumber -Record number of the record to be removed.
		keyNumber    -The key number used to specify whether the key is the
		              primary or one of the alternate keys (-1 for current?).

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*  File Lock Functions
------------------------------------------------------------------------------*/
/*
	PxIOFileLock:
	Lock the specified file.
*/
PXIO_PUBLIC int APIDEF PxIOFileLock(PxIOFileHandle handle);
/*
	Parameters:
		handle -The handle to the file to lock.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOFileUnlock:
	Unlock the specified file.
*/
PXIO_PUBLIC int APIDEF PxIOFileUnlock(PxIOFileHandle handle);
/*
	Parameters:
		handle -The handle to the file to unlock.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIORecordUnlock:
	Unlock a record of the specified file.
*/
PXIO_PUBLIC int APIDEF PxIORecordUnlock(PxIOFileHandle handle);
/*
	Parameters:
		handle -The handle to the file with the record to unlock.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*  File Dictionary Functions
------------------------------------------------------------------------------*/
/*
	PxIOGetDictionaryInfo:
	Returns file dictionary information (if any) for the specified file.
*/
PXIO_PUBLIC int APIDEF PxIOGetDictionaryInfo(PxIOFileHandle handle, char *buffer, size_t *bufferSize, PxIODictionaryInfoType dictionaryInfo);
/*
	Parameters:
		handle         -The handle to the file to retrieve the dictionary info
		                from.
		buffer         -The buffer in which to return the dictionary
		                information.
		bufferSize     -On input the size of the dictionary info buffer, on
		                output the length of the dictionary info or the needed
		                buffer size if there was a length error.
		dictionaryInfo -The dictionary info requested.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIODictionaryRead:
	Read record from embedded data dictionary of the specified file.
*/
PXIO_PUBLIC int APIDEF PxIODictionaryRead(PxIOFileHandle handle, int dictionaryIndex, char *dictionaryBuffer, size_t *dictionaryBufferSize);
/*
	Parameters:
		handle               -The handle to the file to read the dictionary
		                      record from.
		dictionaryIndex      -The index number of the dictionary record to
		                      read.
		dictionaryBuffer     -Storage to save the dictionary record read.
		dictionaryBufferSize -On input the size of the dictionary info buffer, on
		                      output the length of the dictionary info or the needed
		                      buffer size if there was a length error.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIODictionaryWrite:
	Add/Update a record to the dictionary for the specified file.
*/
PXIO_PUBLIC int APIDEF PxIODictionaryWrite(PxIOFileHandle handle, int dictionaryIndex, const char *dictionaryRecord, size_t dictionaryRecordSize);
/*
	Parameters:
		handle               -The handle to the file to perform the write on.
		dictionaryIndex      -The index number of the dictionary record to
		                      write.
		dictionaryRecord     -The buffer with the dictionary record to write
		                      to the file.
		dictionaryRecordSize -The size of the dictionary record to write.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIODictionaryRemove:
	Remove a record from the dictionary of the specified file.
*/
PXIO_PUBLIC int APIDEF PxIODictionaryRemove(PxIOFileHandle handle, int dictionaryIndex);
/*
	Parameters:
		handle          -The handle to the file to perform the remove on.
		dictionaryIndex -The index number of the dictionary record to remove.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*  File Maintenance Functions
------------------------------------------------------------------------------*/
/*
	PxIOFileCreate:
	Create a PxPlus file locally or remotely depending on how the service was
	created.
*/
PXIO_PUBLIC int APIDEF PxIOFileCreate(PxIOService service, const char *pathname, PxIOFileType type, unsigned int recordSize,
	unsigned int keySize, unsigned int maxRecordNumber, const char *keyDefinitions, unsigned int options,
	unsigned int blockSize, unsigned char fieldSeparator);
/*
	Parameters:
		service         -The PxIO service to use to create the file.
		pathname        -The pathname of the file to create.
		type            -The type of PxPlus file to create.
		recordSize      -The maximum size of a record in the new file.
		keySize         -The size of a key in the new file.
		maxRecordNumber -Maximum number or records allowed in the new file, use
		                 0 for unlimted.
		keyDefintions   -Pointer to buffer containing key definitions for file
		                 being allocated.
		options         -File creation options. See earlier defined options
		                 starting with CREATE_OPTION_COMPRESS.
		blockSize       -The block size of the new file in kilobytes.
		fieldSeparator  -The field separator used by the new file.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOErase:
	Erase the specified file.
*/
PXIO_PUBLIC int APIDEF PxIOErase(PxIOService service, const char *pathname);
/*
	Parameters:
		service         -The PxIO service to use to erase the file.
		pathname        -The pathname of the file to erase.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOPurge:
	Delete contents of the specified file.
*/
PXIO_PUBLIC int APIDEF PxIOPurge(PxIOFileHandle handle);
/*
	Parameters:
		handle -The handle to the file to delete the contents of.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOGetFileTime:
	Retrieve the file's timestamp.
*/
PXIO_PUBLIC int APIDEF PxIOGetFileTime(PxIOFileHandle handle, int64_t *timestamp, PxIOFileTimeType fileTimeType);
/*
	Parameters:
		handle       -The handle to the file with the timestamp to retrieve.
		timestamp    -A pointer to an int64_t that will contain the requested timestamp in POSIX time format.
		fileTimeType -The file timestamp to retrieve.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOSetFileTime:
	Set the file timestamp.
*/
PXIO_PUBLIC int APIDEF PxIOSetFileTime(PxIOFileHandle handle, const int64_t timestamp, PxIOFileTimeType fileTimeType);
/*
	Parameters:
		handle       -The handle to the file with the timestamp to set.
		timestamp    -An int64_t value containing the new timestamp in POSIX time format.
		fileTimeType -What file timestamp to set.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*  File Password Functions
------------------------------------------------------------------------------*/
/*
	PxIOPasswordSet:
	Set the file password.
*/
PXIO_PUBLIC int APIDEF PxIOPasswordSet(PxIOFileHandle handle, const char *password, size_t passwordSize, unsigned char permission);
/*
	Parameters:
		handle       -The handle to the file with the password to set.
		password     -The password that will be set.
		passwordSize -The size of the password that will be set.
		permission   -File permission this password will provide.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOPasswordCopy:
	Copy a file password from one file to another.
*/
PXIO_PUBLIC int APIDEF PxIOPasswordCopy(PxIOFileHandle handleTo, PxIOFileHandle handleFrom);
/*
	Parameters:
		handleTo   -The handle to the file where the password is to be copied.
		handleFrom -The handle to the file with the password to copy.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOPasswordRemove:
	Remove the file password.
*/
PXIO_PUBLIC int APIDEF PxIOPasswordRemove(PxIOFileHandle handle);
/*
	Parameters:
		handle -The handle to the file with the password to remove.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*  Miscellaneous File Functions
------------------------------------------------------------------------------*/
/*
	PxIOFlush:
	Flushes cached writes of the specified file to disk.
*/
PXIO_PUBLIC int APIDEF PxIOFlush(PxIOFileHandle handle);
/*
	Parameters:
		handle -The handle to the file to flush.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/
/*
	PxIOExecuteDirection:
	Sets the direction the PxIORead command will use to read records.
*/
PXIO_PUBLIC int APIDEF PxIOExecuteDirection(PxIOFileHandle handle, int direction);
/*
	Parameters:
		handle    -The handle to the file to set the direction for.
		direction -The number of records forwards(+) or backwards(-) the
		           PxIORead should move when selecting the next record to
		           read.

	Return:
		A success status on success, an error status on failure.

------------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif

#endif /* _PXIO_H_ */
